# -*- coding: utf-8 -*-

from typing import Optional, Callable, Iterable

import xbmc
import xbmcgui
import xbmcplugin
from urllib import parse
# noinspection PyUnresolvedReferences
from inputstreamhelper import Helper as InputStreamHelper

from .provider import Provider
from .providers import Providers
from .resources import Resources
from .source_selector import SourceSelector
from .model import Movie
from .search_store import SearchStore


class ContextMenu:
    def __init__(self, addon_url: str):
        self.__menu = []
        self.__addon_url = addon_url

    def __make_url(self, **kwargs) -> str:
        return '{0}?{1}'.format(self.__addon_url, parse.urlencode(kwargs))

    def add_if(self, condition: bool, text: str, **kwargs) -> None:
        if condition:
            self.__menu.append(
                (text, 'RunPlugin({0})'.format(self.__make_url(**kwargs)))
            )

    def add_to_list_item(self, list_item: xbmcgui.ListItem) -> None:
        list_item.addContextMenuItems(self.__menu)


class Actions:
    def __init__(self, providers: Providers, source_selector: SourceSelector, resources: Resources) -> None:
        self.__providers = providers
        self.__addon_handle = resources.addon_handle()
        self.__addon_url = resources.addon_url()
        self.__source_selector = source_selector
        self.__resources = resources

    def list_providers(self) -> None:
        xbmcplugin.setPluginCategory(self.__addon_handle, '')
        xbmcplugin.setContent(self.__addon_handle, 'movies')

        for provider_id in self.__providers.get_ids():
            provider = self.__providers.get(provider_id)

            list_item = xbmcgui.ListItem(label=provider.get_name(), offscreen=True)
            list_item.setProperty('IsPlayable', 'false')
            list_item.setInfo('video', provider.get_info())
            list_item.setArt(provider.get_art())

            url = self.__make_url(action='movies', provider=provider_id)
            xbmcplugin.addDirectoryItem(self.__addon_handle, url, list_item, True)

        xbmcplugin.addSortMethod(self.__addon_handle, xbmcplugin.SORT_METHOD_VIDEO_TITLE)
        xbmcplugin.endOfDirectory(self.__addon_handle)

    @staticmethod
    def __merge_art(art1: dict, art2: dict) -> dict:
        result = art1.copy()
        result.update({k: v for k, v in art2.items() if v is not None})
        return result

    def list_movies(self, provider_id: str, dir_id: Optional[str]) -> None:
        provider = self.__providers.get(provider_id)
        xbmcplugin.setPluginCategory(self.__addon_handle, provider.get_name())
        xbmcplugin.setContent(self.__addon_handle, 'movies')

        if dir_id is None and provider.is_searchable():
            list_item = xbmcgui.ListItem(label=self.__resources.translate('SEARCH', '[Search]'), offscreen=True)
            xbmcplugin.addDirectoryItem(
                self.__addon_handle,
                self.__make_url(action='search', provider=provider_id),
                list_item,
                True
            )

        self.__list_movies(provider, lambda: provider.get_items(dir_id), dir_id)

        xbmcplugin.addSortMethod(self.__addon_handle, xbmcplugin.SORT_METHOD_NONE)
        xbmcplugin.addSortMethod(self.__addon_handle, xbmcplugin.SORT_METHOD_LABEL_IGNORE_THE)
        xbmcplugin.addSortMethod(self.__addon_handle, xbmcplugin.SORT_METHOD_VIDEO_YEAR)
        xbmcplugin.endOfDirectory(self.__addon_handle)

    def search(self, provider_id: str):
        xbmcplugin.setPluginCategory(self.__addon_handle, self.__resources.translate('SEARCH', '[Search]'))
        xbmcplugin.setContent(self.__addon_handle, 'movies')

        list_item = xbmcgui.ListItem(label=self.__resources.translate('NEW_SEARCH', 'New search...'), offscreen=True)
        xbmcplugin.addDirectoryItem(
            self.__addon_handle,
            self.__make_url(action='new_search', provider=provider_id),
            list_item,
            False
        )

        for phrase in SearchStore(provider_id, self.__resources.fs()).get_all():
            list_item = xbmcgui.ListItem(label=phrase, offscreen=True)

            context_menu = ContextMenu(self.__addon_url)
            context_menu.add_if(
                True,
                self.__resources.translate('DELETE_SEARCH', 'Delete search phrase'),
                action='delete_search',
                provider=provider_id,
                phrase=phrase
            )
            context_menu.add_if(
                True,
                self.__resources.translate('EDIT_SEARCH', 'Edit search phrase'),
                action='edit_search',
                provider=provider_id,
                phrase=phrase
            )
            context_menu.add_to_list_item(list_item)

            xbmcplugin.addDirectoryItem(
                self.__addon_handle,
                self.__make_url(action='do_search', provider=provider_id, phrase=phrase),
                list_item,
                True
            )

        xbmcplugin.endOfDirectory(self.__addon_handle)

    def new_search(self, provider_id: str):
        keyboard = xbmc.Keyboard('', self.__resources.translate('ENTER_SEARCH_PHRASE', 'Enter search phrase'), False)
        keyboard.doModal()
        if not keyboard.isConfirmed():
            return
        phrase = keyboard.getText().strip()
        if phrase == '':
            return

        SearchStore(provider_id, self.__resources.fs()).save(phrase)
        self.__resources.gui_helper().open_movie_list(action='do_search', provider=provider_id, phrase=phrase)

    def delete_search(self, provider_id: str, phrase: str):
        SearchStore(provider_id, self.__resources.fs()).delete(phrase)
        self.__resources.gui_helper().refresh()

    def edit_search(self, provider_id: str, phrase: str):
        keyboard = xbmc.Keyboard(
            phrase,
            self.__resources.translate('ENTER_SEARCH_PHRASE', 'Enter search phrase'),
            False
        )

        keyboard.doModal()
        if not keyboard.isConfirmed():
            return

        new_phrase = keyboard.getText().strip()
        if new_phrase == '' or new_phrase == phrase:
            return

        store = SearchStore(provider_id, self.__resources.fs())
        store.replace(phrase, new_phrase)

        self.__resources.gui_helper().open_movie_list(action='do_search', provider=provider_id, phrase=new_phrase)

    def do_search(self, provider_id: str, phrase: str):
        provider = self.__providers.get(provider_id)
        xbmcplugin.setPluginCategory(self.__addon_handle, provider.get_name())
        xbmcplugin.setContent(self.__addon_handle, 'movies')

        self.__list_movies(provider, lambda: provider.search(phrase))

        xbmcplugin.addSortMethod(self.__addon_handle, xbmcplugin.SORT_METHOD_NONE)
        xbmcplugin.addSortMethod(self.__addon_handle, xbmcplugin.SORT_METHOD_LABEL_IGNORE_THE)
        xbmcplugin.addSortMethod(self.__addon_handle, xbmcplugin.SORT_METHOD_VIDEO_YEAR)
        xbmcplugin.endOfDirectory(self.__addon_handle)

    def __list_movies(
            self,
            provider: Provider,
            provider_callable: Callable[[], Iterable[Movie]],
            dir_id: Optional[str] = None
    ) -> None:
        for movie in provider_callable():
            if movie.dir_id is None and dir_id is not None:
                movie.set_dir(dir_id)

            list_item = xbmcgui.ListItem(label=str(movie.title), offscreen=True)

            info = {'title': str(movie.title)}
            info.update(movie.info)
            list_item.setInfo('video', info)
            list_item.setArt(self.__merge_art(provider.get_art(), movie.art))
            list_item.setProperties(movie.properties)

            context_menu = ContextMenu(self.__addon_url)
            context_menu.add_if(
                movie.is_playable,
                self.__resources.translate('SHOW_ALL_SOURCES', 'Show all sources'),
                action='sources_context',
                provider=provider.get_id(),
                dir_id=movie.dir_id,
                id=movie.id
            )
            context_menu.add_to_list_item(list_item)

            is_dir = False
            if movie.is_playable:
                url = self.__make_url(action='play', provider=movie.provider, dir_id=movie.dir_id, id=movie.id)
            elif movie.action is not None:
                params = {
                    'action': 'plugin',
                    'provider': movie.provider,
                    'dir_id': movie.dir_id,
                    'id': movie.id,
                }
                params.update(movie.action)
                url = self.__make_url(**params)
            else:
                is_dir = True
                url = self.__make_url(action='movies', provider=movie.provider, dir_id=movie.id)

            list_item.setProperty('IsPlayable', 'true' if movie.is_playable else 'false')
            xbmcplugin.addDirectoryItem(self.__addon_handle, url, list_item, is_dir)

    def list_sources(self, provider_id: str, dir_id: str, movie_id: str) -> None:
        provider = self.__providers.get(provider_id)
        movie = provider.get_item(dir_id, movie_id)
        playable = provider.get_playable(movie)

        xbmcplugin.setPluginCategory(self.__addon_handle, movie.title)
        xbmcplugin.setContent(self.__addon_handle, 'movies')

        for source in self.__source_selector.supported_sources(playable.sources):
            list_item = xbmcgui.ListItem(label=str(source), offscreen=True)
            movie.info['title'] = str(source)
            list_item.setInfo('video', movie.info)
            list_item.setArt(self.__merge_art(provider.get_art(), movie.art))
            list_item.setProperty('IsPlayable', 'true')

            url = self.__make_url(action='play', provider=movie.provider, dir_id=dir_id, id=movie.id, source=source.id)
            xbmcplugin.addDirectoryItem(self.__addon_handle, url, list_item, False)

        for label, command in playable.commands:
            list_item = xbmcgui.ListItem(label=label, offscreen=True)
            movie.info['title'] = label
            list_item.setInfo('video', movie.info)
            list_item.setArt(self.__merge_art(provider.get_art(), movie.art))
            list_item.setProperty('IsPlayable', 'false')
            url = self.__make_url(**command) if isinstance(command, dict) else command
            xbmcplugin.addDirectoryItem(self.__addon_handle, url, list_item, False)

        xbmcplugin.addSortMethod(self.__addon_handle, xbmcplugin.SORT_METHOD_NONE)
        xbmcplugin.endOfDirectory(self.__addon_handle)

    def play_movie(self, provider_id: str, dir_id: str, movie_id: str, source_id: Optional[str] = None):
        provider = self.__providers.get(provider_id)
        movie = provider.get_item(dir_id, movie_id)
        playable = provider.get_playable(movie)

        source = self.__source_selector.best_source(playable.sources) \
            if source_id is None \
            else self.__source_selector.source_by_id(playable.sources, source_id)

        if source is None:
            xbmcgui.Dialog().notification(
                self.__resources.addon_info('name'),
                self.__resources.translate('NO_PLAYABLE_SOURCES_FOUND', 'No playable sources found.'),
                xbmcgui.NOTIFICATION_WARNING,
                5000
            )
            return

        play_item = xbmcgui.ListItem(path=source.url, offscreen=True)
        play_item.setInfo('video', movie.info)
        play_item.setArt(movie.art)
        play_item.setSubtitles(playable.subtitles)

        if source.manifest is not None:
            try:
                ia_helper = InputStreamHelper(source.manifest.type, drm=source.manifest.license_type)
                if ia_helper.check_inputstream():
                    play_item.setContentLookup(False)
                    play_item.setMimeType(source.manifest.content_type)
                    play_item.setProperty('inputstream', 'inputstream.adaptive')
                    play_item.setProperty('inputstream.adaptive.manifest_type', source.manifest.type)
                    play_item.setProperty('inputstream.adaptive.license_type', source.manifest.license_type)
                    play_item.setProperty('inputstream.adaptive.license_key', source.manifest.license_key)
                    if source.manifest.license_data is not None:
                        play_item.setProperty('inputstream.adaptive.license_data', source.manifest.license_data)

            except Exception as e:
                xbmcgui.Dialog().notification(
                    self.__resources.addon_info('name'),
                    str(e),
                    xbmcgui.NOTIFICATION_ERROR, 5000
                )
                raise e

        xbmcgui.Dialog().notification(self.__resources.addon_info('name'), str(source), xbmcgui.NOTIFICATION_INFO, 5000)
        xbmcplugin.setResolvedUrl(self.__addon_handle, True, listitem=play_item)

    def __make_url(self, **kwargs) -> str:
        return '{0}?{1}'.format(self.__addon_url, parse.urlencode(kwargs))
