# -*- coding: utf-8 -*-
from __future__ import annotations

from time import sleep
from typing import Optional, Dict

import simplejson as json
# noinspection PyProtectedMember
from bs4 import BeautifulSoup, SoupStrainer
from requests import Request, RequestException, Session, Response

from .logger import Logger


class HttpClient:
    def __init__(
            self,
            logger: Logger,
            session: Session,
            debug: bool,
            user_agent: str,
            timeout_seconds: int = 10,
            retries: int = 5,
            retry_delay_seconds: int = 3
    ) -> None:
        self.__logger = logger
        self.__session = session
        self.__timeout_seconds = timeout_seconds
        self.__retries = retries
        self.__retry_delay_seconds = retry_delay_seconds
        self.__debug = debug
        self.__headers = {
            'accept': '*/*',
            'user-agent': user_agent
        }

    def fetch(
            self,
            url: str,
            method: str = 'GET',
            referer: Optional[str] = None,
            headers: Optional[Dict[str, str]] = None,
            data: Optional[str] = None,
            params: Optional[Dict[str, str]] = None,
            cookies: Optional[Dict[str, str]] = None,
            allow_redirects: bool = True
    ) -> Response:
        return self.__make_request(url, method, referer, headers, data, params, cookies, allow_redirects)

    def fetch_text(
            self,
            url: str,
            method: str = 'GET',
            referer: Optional[str] = None,
            headers: Optional[Dict[str, str]] = None,
            data: Optional[str] = None,
            params: Optional[Dict[str, str]] = None,
            cookies: Optional[Dict[str, str]] = None,
    ) -> str:
        return self.__make_request(url, method, referer, headers, data, params, cookies).text

    def fetch_json(
            self,
            url: str,
            method: str = 'GET',
            referer: Optional[str] = None,
            headers: Optional[Dict[str, str]] = None,
            data: Optional[str] = None,
            params: Optional[Dict[str, str]] = None,
            cookies: Optional[Dict[str, str]] = None,
    ) -> dict:
        return self.__make_request(url, method, referer, headers, data, params, cookies).json()

    def fetch_jsonp(
            self,
            url: str,
            method: str = 'GET',
            referer: Optional[str] = None,
            headers: Optional[Dict[str, str]] = None,
            data: Optional[str] = None,
            params: Optional[Dict[str, str]] = None,
            cookies: Optional[Dict[str, str]] = None,
            arg_num: int = 0
    ) -> dict:
        text = self.fetch_text(url, method, referer, headers, data, params, cookies)
        start = text.find('(') + 1
        end = text.rfind(')')
        return json.loads('[' + text[start:end].strip() + ']')[arg_num]

    def fetch_soup(
            self,
            url: str,
            method: str = 'GET',
            referer: Optional[str] = None,
            headers: Optional[Dict[str, str]] = None,
            data: Optional[str] = None,
            params: Optional[Dict[str, str]] = None,
            cookies: Optional[Dict[str, str]] = None,
            parse_only: Optional[SoupStrainer] = None
    ) -> BeautifulSoup:
        text = self.fetch_text(url, method, referer, headers, data, params, cookies)
        return BeautifulSoup(text, features='html.parser', parse_only=parse_only)

    def __is_retryable(self, e: RequestException, attempt_number: int) -> bool:
        # do not retry client errors
        try:
            if 400 <= e.response.status_code < 500:
                return False
        except AttributeError:
            pass

        # retry only if retry limit is not reached yet
        return attempt_number < self.__retries

    def __make_request(
            self,
            url: str,
            method: str = 'GET',
            referer: Optional[str] = None,
            headers: Optional[Dict[str, str]] = None,
            data: Optional[str] = None,
            params: Optional[Dict[str, str]] = None,
            cookies: Optional[Dict[str, str]] = None,
            allow_redirects: bool = True
    ):
        attempt_number = 0

        req_headers = self.__headers
        if referer:
            req_headers['referer'] = referer

        if headers is not None:
            req_headers.update(headers)

        req = Request(method=method, url=url, headers=req_headers, data=data, params=params, cookies=cookies)

        while attempt_number < self.__retries:
            try:
                resp = self.__session.send(
                    req.prepare(),
                    timeout=self.__timeout_seconds,
                    allow_redirects=allow_redirects
                )
                if resp.status_code == 404:
                    self.__logger.warning('Request "{0}" "{1}" returns 404.', method, url)
                    raise HttpNotFound(req, resp)
                resp.raise_for_status()

                if self.__debug:
                    # noinspection PyUnresolvedReferences
                    from_cache = resp.from_cache if hasattr(resp, 'from_cache') else False
                    self.__logger.notice(
                        '{0} "{1}" "{2}" (ref: "{3}", data: "{4}", time: "{5}").',
                        'Cached request' if from_cache else 'Request',
                        method,
                        url,
                        req_headers['referer'] if 'referer' in req_headers else 'None',
                        self.__censor_log_data(data if data is not None else params),
                        resp.elapsed
                    )

                return resp
            except RequestException as e:
                if self.__is_retryable(e, attempt_number):
                    self.__logger.warning(
                        'Request "{0}" "{1}" failed: "{2}". Retrying after {3}s.',
                        method,
                        url,
                        type(e).__name__,
                        self.__retry_delay_seconds
                    )
                    sleep(self.__retry_delay_seconds)
                    attempt_number += 1
                    continue
                self.__logger.error(
                    'Request "{0}" "{1}" failed: "{2}". Giving up.',
                    method,
                    url,
                    type(e).__name__,
                )
                raise e

    @staticmethod
    def __censor_log_data(data):
        try:
            if isinstance(data, str):
                data = json.loads(data)
        except ValueError:
            pass

        if isinstance(data, dict):
            censored = dict(data)
            for censored_key in ['password', 'Password']:
                if censored_key in censored and censored[censored_key] != '':
                    censored[censored_key] = '***'
            return censored

        return data


class HttpNotFound(Exception):
    def __init__(self, request: Request, response: Response) -> None:
        self.message = 'Page {0} was not found.'.format(request.url)
        self.request = request
        self.response = response
