# -*- coding: utf-8 -*-
from abc import ABCMeta, abstractmethod
from typing import final, Dict

from requests import RequestException
from xbmcaddon import Addon
from xbmcgui import Dialog, NOTIFICATION_ERROR

from .actions import Actions
from .config_actions import ConfigActions
from .http_client import HttpNotFound
from .profiler import Profiler
from .provider import ProviderException, Provider
from .providers import Providers
from .resources import Resources
from .source_selector import SourceSelector


class Main:
    __metaclass__ = ABCMeta

    def __init__(self, addon: Addon, argv: list, translations: Dict[str, int]) -> None:
        self.__resources = Resources(addon, translations, argv)
        self.__providers = Providers()

        self.setup(self.__resources)

    @abstractmethod
    def setup(self, resources: Resources) -> None:
        pass

    @final
    def run(self) -> None:
        try:
            with Profiler(self.__resources.addon_info('name') + ':main', self.__resources):
                self.__main(self.__resources.addon_params())
        except (HttpNotFound, RequestException, ProviderException) as e:
            self.__resources.logger().error(str(e))
            Dialog().notification(self.__resources.addon_info('name'), str(e), NOTIFICATION_ERROR, 5000)

    @final
    def add_provider(self, provider: Provider) -> None:
        self.__providers.add_provider(provider)

    def __main(self, params: dict) -> None:
        def actions() -> Actions:
            return Actions(self.__providers, SourceSelector(), self.__resources)

        def config_actions() -> ConfigActions:
            return ConfigActions(self.__resources)

        def plugin_action(provider_id: str, parameters: dict) -> None:
            self.__providers.get(provider_id).action(parameters)

        def p(name: str) -> str:
            return None if name not in params or params[name] == 'None' else params[name]

        action = p('action')

        if action == 'play':
            actions().play_movie(p('provider'), p('dir_id'), p('id'), p('source'))

        elif action == 'sources':
            actions().list_sources(p('provider'), p('dir_id'), p('id'))

        elif action == 'sources_context':
            self.__resources.gui_helper().open_movie_list(
                action='sources',
                provider=p('provider'),
                dir_id=p('dir_id'),
                id=p('id')
            )

        elif action == 'movies':
            actions().list_movies(p('provider'), p('dir_id'))

        elif action == 'clear_requests_cache':
            config_actions().clear_requests_cache(p('provider'))

        elif action == 'clear_session':
            config_actions().clear_session(p('provider'))

        elif action == 'open_settings':
            config_actions().open_settings()

        elif action == 'plugin':
            plugin_action(p('provider'), params)

        elif action == 'search':
            actions().search(p('provider'))

        elif action == 'new_search':
            actions().new_search(p('provider'))

        elif action == 'do_search':
            actions().do_search(p('provider'), p('phrase'))

        elif action == 'delete_search':
            actions().delete_search(p('provider'), p('phrase'))

        elif action == 'edit_search':
            actions().edit_search(p('provider'), p('phrase'))

        elif self.__providers.count() == 1:
            actions().list_movies(self.__providers.get_ids().pop(), None)

        else:
            actions().list_providers()
