# -*- coding: utf-8 -*-

from __future__ import annotations
from typing import Optional, Union, List, Tuple


class Movie:
    """Represents movie / series episode"""
    def __init__(self, movie_id: str, provider: str, title: str):
        self.id: str = movie_id
        self.provider: str = provider
        self.title: str = title
        self.info: dict = {}
        self.art: dict = {}
        self.properties: dict = {}
        self.is_playable: bool = True
        self.meta_data: dict = {}
        self.dir_id: Optional[str] = None
        self.action: Optional[dict] = None

    def set_info(self, info: Info) -> Movie:
        self.info = info.to_dict()
        return self

    def set_art(self, art: Art) -> Movie:
        self.art = art.to_dict()
        return self

    def set_property(self, name: str, value: str) -> Movie:
        self.properties[name] = value
        return self

    def set_total_episodes(self, total_episodes: int) -> Movie:
        return self.set_property('TotalEpisodes', str(total_episodes))

    def set_meta(self, name: str, value) -> Movie:
        self.meta_data[name] = value
        return self

    def set_dir(self, dir_id: str) -> Movie:
        self.dir_id = dir_id
        return self


class MovieDir(Movie):
    """Represents category/tvshow/season or other 'directory' that contains movies"""
    def __init__(self, movie_id: str, provider: str, title: str):
        Movie.__init__(self, movie_id, provider, title)
        self.is_playable = False


class DirAction(Movie):
    """Represents action on the list"""
    def __init__(self, movie_id: str, provider: str, title: str, action: dict = None):
        Movie.__init__(self, movie_id, provider, title)
        self.is_playable = False
        self.action: dict = action if action else {}


class MediaType:
    VIDEO = 'video'
    MOVIE = 'movie'
    TVSHOW = 'tvshow'
    SEASON = 'season'
    EPISODE = 'episode'
    MUSICVIDEO = 'musicvideo'


class Art(object):
    def __init__(
        self,
        thumb: Optional[str] = None,
        poster: Optional[str] = None,
        banner: Optional[str] = None,
        fanart: Optional[str] = None,
        clearart: Optional[str] = None,
        clearlogo: Optional[str] = None,
        landscape: Optional[str] = None,
        icon: Optional[str] = None
    ):
        self.thumb: Optional[str] = thumb
        self.poster: Optional[str] = poster
        self.banner: Optional[str] = banner
        self.fanart: Optional[str] = fanart
        self.clearart: Optional[str] = clearart
        self.clearlogo: Optional[str] = clearlogo
        self.landscape: Optional[str] = landscape
        self.icon: Optional[str] = icon

    def to_dict(self) -> dict:
        return {k: v for k, v in self.__dict__.items() if v is not None}


class Info(object):
    def __init__(self, count: Optional[int] = None, size: Optional[int] = None, date: Optional[str] = None):
        self.count: Optional[int] = count
        self.size: Optional[int] = size
        self.date: Optional[str] = date

    def to_dict(self) -> dict:
        return {k: v for k, v in self.__dict__.items() if v is not None}


class VideoInfo(Info):
    def __init__(
            self,
            mediatype: Optional[MediaType] = None,
            date: Optional[str] = None,
            size: Optional[int] = None,
            count: Optional[int] = None,
            genre: Union[str, List[str], None] = None,
            country: Union[str, List[str], None] = None,
            year: Optional[int] = None,
            episode: Optional[int] = None,
            season: Optional[int] = None,
            sortepisode: Optional[int] = None,
            sortseason: Optional[int] = None,
            episodeguide: Optional[str] = None,
            showlink: Optional[str] = None,
            top250: Optional[int] = None,
            setid: Optional[int] = None,
            tracknumber: Optional[int] = None,
            rating: Optional[float] = None,
            userrating: Optional[int] = None,
            playcount: Optional[int] = None,
            overlay: Optional[int] = None,
            cast: Optional[List[str]] = None,
            castandrole: Optional[List[Tuple[str, str]]] = None,
            director: Union[str, List[str], None] = None,
            mpaa: Optional[str] = None,
            plot: Optional[str] = None,
            plotoutline: Optional[str] = None,
            title: Optional[str] = None,
            originaltitle: Optional[str] = None,
            sorttitle: Optional[str] = None,
            duration: Optional[str] = None,
            studio: Union[str, List[str], None] = None,
            tagline: Optional[str] = None,
            writer: Union[str, List[str], None] = None,
            tvshowtitle: Optional[str] = None,
            premiered: Optional[str] = None,
            status: Optional[str] = None,
            set: Optional[str] = None,
            setoverview: Optional[str] = None,
            tag: Union[str, List[str], None] = None,
            imdbnumber: Optional[str] = None,
            code: Optional[str] = None,
            aired: Optional[str] = None,
            credits: Union[str, List[str], None] = None,
            lastplayed: Optional[str] = None,
            album: Optional[str] = None,
            artist: Optional[str] = None,
            votes: Optional[str] = None,
            path: Optional[str] = None,
            trailer: Optional[str] = None,
            dateadded: Optional[str] = None,
    ):
        Info.__init__(self, count=count, size=size, date=date)

        self.mediatype = mediatype
        self.genre = genre
        self.country = country
        self.year = year
        self.episode = episode
        self.season = season
        self.sortepisode = sortepisode
        self.sortseason = sortseason
        self.episodeguide = episodeguide
        self.showlink = showlink
        self.top250 = top250
        self.setid = setid
        self.tracknumber = tracknumber
        self.rating = rating
        self.userrating = userrating
        self.playcount = playcount
        self.overlay = overlay
        self.cast = cast
        self.castandrole = castandrole
        self.director = director
        self.mpaa = mpaa
        self.plot = plot
        self.plotoutline = plotoutline
        self.title = title
        self.originaltitle = originaltitle
        self.sorttitle = sorttitle
        self.duration = duration
        self.studio = studio
        self.tagline = tagline
        self.writer = writer
        self.tvshowtitle = tvshowtitle
        self.premiered = premiered
        self.status = status
        self.set = set
        self.setoverview = setoverview
        self.tag = tag
        self.imdbnumber = imdbnumber
        self.code = code
        self.aired = aired
        self.credits = credits
        self.lastplayed = lastplayed
        self.album = album
        self.artist = artist
        self.votes = votes
        self.path = path
        self.trailer = trailer
        self.dateadded = dateadded


class Playable:
    def __init__(self):
        self.sources = []
        self.subtitles = []
        self.commands = []

    def add_source(self, source: Source) -> Playable:
        self.sources.append(source)
        return self

    def add_subtitle(self, subtitle_url: str) -> Playable:
        self.subtitles.append(subtitle_url)
        return self

    def add_command(self, label: str, command: Optional[dict] = None):
        self.commands.append((label, command))


class Source:
    def __init__(
            self,
            source_id: str,
            url: str,
            adaptive: bool = False,
            video_resolution: Optional[int] = None,
            video_bitrate: Optional[int] = None,
            audio_bitrate: Optional[int] = None
    ):
        self.id = source_id
        self.url = url
        self.video_resolution = video_resolution
        self.video_bitrate = video_bitrate
        self.audio_bitrate = audio_bitrate
        self.adaptive = adaptive
        self.manifest = None
        self.priority = 0
        self.description = None

    def set_manifest(self, manifest: Manifest) -> Source:
        self.manifest = manifest
        return self

    # custom priority, for sorting sources
    def set_priority(self, priority: int) -> Source:
        self.priority = priority
        return self

    def set_description(self, description: str) -> Source:
        self.description = description
        return self

    def __str__(self) -> str:
        info = ''
        if self.video_resolution:
            info += '{0} / '.format(self.video_resolution)

        info += str(self.manifest or 'Stream')

        if self.video_bitrate:
            info += ' / video {0}k'.format(self.video_bitrate)
        if self.audio_bitrate:
            info += ' / audio {0}k'.format(self.audio_bitrate)

        if self.description:
            info += ' / ' + self.description

        return info


class Manifest:
    def __init__(
            self,
            manifest_type: str,
            content_type: Optional[str] = None,
            license_type: Optional[str] = None,
            license_key: Optional[str] = None,
            license_data: Optional[str] = None
    ):
        self.type = manifest_type
        self.content_type = content_type
        self.license_type = license_type
        self.license_key = license_key
        self.license_data = license_data
        self.supported_platforms = ['windows', 'linux', 'android']

    def __str__(self) -> str:
        return '{0} / {1}'.format(self.type, self.license_type)


class MpdManifest(Manifest):
    def __init__(self, license_key: Optional[str] = None, license_data: Optional[str] = None):
        Manifest.__init__(self, 'mpd', 'application/dash+xml', 'com.widevine.alpha', license_key, license_data)

    def __str__(self) -> str:
        if self.license_key is None:
            return 'MPEG-DASH'
        return 'MPEG-DASH / Widevine'


class IsmManifest(Manifest):
    def __init__(self):
        Manifest.__init__(self, 'ism', 'application/vnd.ms-sstr+xml')

    def __str__(self) -> str:
        return 'ISM'


class IsmWidevineManifest(Manifest):
    def __init__(self, license_key: Optional[str] = None, license_data: Optional[str] = None):
        Manifest.__init__(self, 'ism', 'video/mp4', 'com.widevine.alpha', license_key, license_data)

    def __str__(self) -> str:
        if self.license_key is None:
            return 'MP4'
        return 'MP4 / Widevine'


class IsmPlayReadyManifest(Manifest):
    def __init__(self):
        Manifest.__init__(self, 'ism', 'application/vnd.ms-sstr+xml', 'com.microsoft.playready')
        self.supported_platforms = ['android']

    def __str__(self) -> str:
        return 'ISM / PlayReady'


class HlsManifest(Manifest):
    def __init__(self, license_key: Optional[str] = None):
        Manifest.__init__(self, 'hls', 'application/vnd.apple.mpegurl', None, license_key)

    def __str__(self) -> str:
        return 'HLS'
