# -*- coding: utf-8 -*-

from abc import ABCMeta, abstractmethod
from os import path
from typing import Optional, Dict, Iterator

from .model import Movie, Playable


class ProviderException(Exception):
    pass


class Provider:
    __metaclass__ = ABCMeta

    @abstractmethod
    def get_id(self) -> str:
        pass

    @abstractmethod
    def get_name(self) -> str:
        pass

    def get_info(self) -> Dict[str, str]:
        return {
            'title': self.get_name()
        }

    def get_art(self) -> Dict[str, str]:
        return {}

    @abstractmethod
    def get_items(self, dir_id: Optional[str]) -> Iterator[Movie]:
        pass

    def get_item(self, dir_id: str, movie_id: str) -> Movie:
        # default implementation is possibly inefficient, providers can override it
        movies = self.get_items(dir_id)
        return next((movie for movie in movies if str(movie.id) == str(movie_id)), None)

    @abstractmethod
    def get_playable(self, movie: Movie) -> Playable:
        pass

    def action(self, params: dict) -> None:
        pass

    # noinspection PyMethodMayBeStatic
    def is_searchable(self) -> bool:
        return False

    def search(self, phrase: str) -> Iterator[Movie]:
        pass

    @staticmethod
    def _get_art(base_path: str) -> Dict[str, str]:
        dir_path = path.dirname(path.abspath(base_path))
        return {
            'thumb': path.join(dir_path, 'icon.png'),
            'fanart': path.join(dir_path, 'fanart.jpg')
        }
