# -*- coding: utf-8 -*-

import re
from os import path
from shutil import rmtree

from typing import Optional, Union, Dict
from urllib import parse

import requests
import simplejson as json
from xbmc import executebuiltin
from xbmcaddon import Addon
from xbmcvfs import mkdirs, translatePath

from .http_client import HttpClient
from .logger import Logger
from .vendor.cachecontrol import CacheControl
from .vendor.cachecontrol.heuristics import BaseHeuristic, LastModified
from .vendor.diskcache import Cache


class DebugSettings:
    def __init__(self, profiling: bool, debug_requests: bool) -> None:
        self.profiling = profiling
        self.requests = debug_requests


class Session:
    def __init__(self, addon: Addon):
        self.__addon = addon

    def set(self, provider_id: str, data: dict) -> None:
        self.__addon.setSettingString('session_' + provider_id, json.dumps(data, encoding='utf-8'))

    def get(self, provider_id: str) -> dict:
        try:
            raw_session = self.__addon.getSettingString('session_' + provider_id)
        except TypeError:
            raw_session = None

        if raw_session is None or raw_session == '':
            return {}
        session = json.loads(raw_session, encoding='utf-8')
        if not isinstance(session, dict):
            return {}
        return session

    def clear(self, provider_id: str) -> None:
        self.__addon.setSettingString('session_' + provider_id, json.dumps({}))


class GuiHelper:
    def __init__(self, plugin_url: str):
        self.__plugin_url = plugin_url

    @staticmethod
    def refresh():
        executebuiltin("Container.Refresh")

    def open_movie_list(self, **kwargs) -> None:
        executebuiltin("ActivateWindow(10025,{0}?{1},return)".format(self.__plugin_url, parse.urlencode(kwargs)))


class Settings:
    def __init__(self, addon: Addon, gui_helper: GuiHelper):
        self.__addon = addon
        self.__gui_helper = gui_helper

    def open(self) -> None:
        self.__addon.openSettings()
        self.__gui_helper.refresh()

    def get(self, option: str) -> Union[str, int, bool, None]:
        try:
            value = self.__addon.getSettingString(option)
        except TypeError:
            try:
                value = self.__addon.getSettingInt(option)
            except TypeError:
                try:
                    value = self.__addon.getSettingBool(option)
                except TypeError:
                    value = None
        return value


class PluginDataDir:

    def __init__(self, addon_data_folder: str) -> None:
        self.__addon_data_folder = addon_data_folder
        mkdirs(self.__addon_data_folder)

    @staticmethod
    def __make_safe_file(s: str) -> str:
        s = str(s).strip().replace(' ', '_')
        return re.sub(r'(?u)[^-\w.]', '', s)

    def get_path(self, *file_path_elements) -> str:
        file_path_elements = map(
            lambda x: self.__make_safe_file(x),
            file_path_elements
        )
        return translatePath(path.join(self.__addon_data_folder, *file_path_elements))

    def file_exists(self, *file_path_elements) -> bool:
        file_name = self.get_path(*file_path_elements)
        return path.exists(file_name)

    def dir_unlink(self, *file_path_elements) -> None:
        if not self.file_exists(*file_path_elements):
            return

        rmtree(self.get_path(*file_path_elements))


class Resources:
    def __init__(self, addon: Addon, translations: Dict[str, int], argv: list):
        self.__addon_params = dict(parse.parse_qsl(argv[2][1:]))
        self.__addon_handle = int(argv[1])
        self.__addon_url = argv[0]

        self.__addon = addon
        self.__logger = Logger(self.__addon.getAddonInfo('id'))
        self.__plugin_data_dir = PluginDataDir(translatePath(self.__addon.getAddonInfo('profile')))
        self.__session = Session(addon)
        self.__gui_helper = GuiHelper(self.__addon_url)
        self.__settings = Settings(addon, self.__gui_helper)
        self.__translation_map = translations

    def addon_params(self) -> dict:
        return self.__addon_params

    def addon_handle(self) -> int:
        return self.__addon_handle

    def addon_url(self) -> str:
        return self.__addon_url

    def addon_info(self, info_id: str) -> str:
        return self.__addon.getAddonInfo(info_id)

    def settings(self) -> Settings:
        return self.__settings

    def session(self) -> Session:
        return self.__session

    @staticmethod
    def user_agent() -> str:
        return 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/92.0.4515.159 '\
               'Safari/537.36 Edg/92.0.902.78'

    def debug(self) -> DebugSettings:
        return DebugSettings(
            self.__addon.getSettingBool('log_profiling'),
            self.__addon.getSettingBool('log_requests')
        )

    def translate(self, tr_id: Union[int, str], default: Optional[str] = None) -> str:
        if tr_id in self.__translation_map:
            tr_id = self.__translation_map[tr_id]

        elif not isinstance(tr_id, int):
            return default if default is not None else 'Untranslated {}'.format(tr_id)

        return self.__addon.getLocalizedString(tr_id)

    def logger(self) -> Logger:
        return self.__logger

    def http_client(
            self,
            provider_id: str,
            heuristic: Optional[BaseHeuristic] = None,
            timeout_seconds: int = 10,
            retries: int = 5,
            retry_delay_seconds: int = 3
    ) -> HttpClient:
        class MyCache(Cache):
            def __bool__(self):
                return True
            __nonzero__ = __bool__

        return HttpClient(
            self.logger(),
            CacheControl(
                requests.Session(),
                cache=MyCache(self.fs().get_path('request-cache-' + provider_id)),
                heuristic=heuristic if heuristic else LastModified(),
                cacheable_methods=('GET',)
            ),
            self.debug().requests,
            self.user_agent(),
            timeout_seconds,
            retries,
            retry_delay_seconds
        )

    def fs(self) -> PluginDataDir:
        return self.__plugin_data_dir

    def gui_helper(self) -> GuiHelper:
        return self.__gui_helper
