# -*- coding: utf-8 -*-

from functools import cmp_to_key
from typing import Optional, Union, List

from xbmc import getCondVisibility

from .model import Source


class SourceSelector:
    def __init__(self) -> None:
        self.__platform = self.__get_platform()

    def best_source(self, sources: List[Source]) -> Optional[Source]:
        return next((source for source in self.supported_sources(sources)), None)

    def supported_sources(self, sources: List[Source]) -> List[Source]:
        return [x for x in sorted(sources, key=cmp_to_key(self.__comparator)) if self.__is_supported(x)]

    @staticmethod
    def source_by_id(sources: List[Source], source_id: str) -> Optional[Source]:
        return next((source for source in sources if source.id == source_id), None)

    @staticmethod
    def __get_platform() -> str:
        if getCondVisibility('System.Platform.Android'):
            return 'android'

        if getCondVisibility('System.Platform.Linux'):
            return 'linux'

        if getCondVisibility('System.Platform.Windows'):
            return 'windows'

        return 'other'

    @staticmethod
    def __num(num: Union[int, str, None]) -> int:
        return 0 if num is None else int(num)

    def __comparator(self, x: Source, y: Source) -> int:
        if x.priority != y.priority:
            return y.priority - x.priority

        if x.adaptive != y.adaptive:
            return 1 if y.adaptive else -1

        if self.__num(x.video_resolution) != self.__num(y.video_resolution):
            return self.__num(y.video_resolution) - self.__num(x.video_resolution)

        if self.__num(x.video_bitrate) != self.__num(y.video_bitrate):
            return self.__num(y.video_bitrate) - self.__num(x.video_bitrate)

        if self.__num(x.audio_bitrate) != self.__num(y.audio_bitrate):
            return self.__num(y.audio_bitrate) - self.__num(x.audio_bitrate)

        return 0

    def __is_supported(self, source: Source) -> bool:
        return source.manifest is None or self.__platform in source.manifest.supported_platforms
